<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Product;
use App\Models\Apishowroom;

class ProductSyncController extends Controller
{
    public function syncProductsToShowroom(Request $request)
    {
        $request->validate([
            'apishowroom_ids' => ['required', 'array', 'min:1'],
            'apishowroom_ids.*' => ['exists:apishowrooms,id'],
        ]);

        $targetApishowroomIds = $request->input('apishowroom_ids');

        $products = Product::orderBy('id')->get()->map(function ($product) {
            return [
                'id' => $product->id,
                'productname' => $product->productname,
                'productiondepartment' => $product->productiondepartment,
                'productcategory' => $product->productcategory,
                'price' => (float) $product->price,
                'dateofentry' => $product->dateofentry 
    ? \Carbon\Carbon::parse($product->dateofentry)->toDateTimeString()
    : null,
            ];
        })->toArray();

        $apishowroomsToSync = Apishowroom::whereIn('id', $targetApishowroomIds)->get();
        $syncResults = [];

        foreach ($apishowroomsToSync as $apishowroom) {
            if ($apishowroom->api_url) {
                try {
                    $http = Http::timeout(60);
                    if ($apishowroom->api_token) {
                        $http->withToken($apishowroom->api_token);
                    }

                    $response = $http->post($apishowroom->api_url, [
                        'products' => $products,

                    ]);
                    Log::debug('Pushed products to ' . $apishowroom->name, ['payload' => $products]);

                    if ($response->successful()) {
                        $syncResults[] = " Synced products to showroom: {$apishowroom->name}";
                        Log::info("Product sync to {$apishowroom->name} succeeded.");
                        $apishowroom->update(['last_pulled_at' => now()]);
                    } else {
                        $syncResults[] = "❌ Failed to sync to showroom: {$apishowroom->name}. Status: {$response->status()}";
                        Log::error("Product sync failed to {$apishowroom->name}: " . $response->body());
                    }
                } catch (\Exception $e) {
                    $syncResults[] = " Exception syncing to {$apishowroom->name}: " . $e->getMessage();
                    Log::error("Exception syncing to {$apishowroom->name}", ['exception' => $e]);
                }
            } else {
                $syncResults[] = " Skipped showroom '{$apishowroom->name}': missing API URL.";
            }
        }

        return response()->json(['sync_results' => $syncResults]);

    }
}
